/*
 * Decompiled with CFR 0.152.
 */
package net.citizensnpcs.api.astar.pathfinder;

import com.google.common.collect.ImmutableList;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinWorkerThread;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.citizensnpcs.api.CitizensAPI;
import net.citizensnpcs.api.ai.NavigatorParameters;
import net.citizensnpcs.api.astar.AStarMachine;
import net.citizensnpcs.api.astar.pathfinder.BlockSource;
import net.citizensnpcs.api.astar.pathfinder.Path;
import net.citizensnpcs.api.astar.pathfinder.VectorGoal;
import net.citizensnpcs.api.astar.pathfinder.VectorNode;
import net.citizensnpcs.api.util.BoundingBox;
import net.citizensnpcs.api.util.Messaging;
import net.citizensnpcs.api.util.SpigotUtil;
import org.bukkit.Bukkit;
import org.bukkit.Chunk;
import org.bukkit.ChunkSnapshot;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.World;
import org.bukkit.block.data.BlockData;
import org.bukkit.plugin.Plugin;

public class AsyncChunkCache {
    private final ScheduledExecutorService evictionExecutor;
    private final Plugin plugin;
    private final Map<ChunkKey, CompletableFuture<ChunkSnapshot>> snapshotCache = new ConcurrentHashMap<ChunkKey, CompletableFuture<ChunkSnapshot>>();
    private final Map<ChunkKey, Long> snapshotCacheExpiry = new ConcurrentHashMap<ChunkKey, Long>();
    private final long ttlMillis;
    private final ForkJoinPool workerPool;
    private static MethodHandle WORLD_GET_CHUNK_AT_ASYNC;
    private static MethodHandle WORLD_GET_CHUNKS_AT_ASYNC;

    public AsyncChunkCache(Plugin plugin, int workerThreads, long cacheTtlMillis) {
        this.plugin = plugin;
        this.workerPool = ForkJoinPool.commonPool();
        this.ttlMillis = cacheTtlMillis;
        if (cacheTtlMillis > 0L) {
            this.evictionExecutor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "Citizens Async Pathfinder Cache Eviction Thread"));
            this.evictionExecutor.scheduleAtFixedRate(this::evictStaleChunks, cacheTtlMillis, cacheTtlMillis, TimeUnit.MILLISECONDS);
        } else {
            this.evictionExecutor = null;
        }
    }

    private void evictStaleChunks() {
        long now = System.currentTimeMillis();
        for (Map.Entry<ChunkKey, Long> e : this.snapshotCacheExpiry.entrySet()) {
            CompletableFuture<ChunkSnapshot> cf;
            ChunkKey key = e.getKey();
            long last = e.getValue();
            if (now <= last || (cf = this.snapshotCache.get(key)) == null || !cf.isDone()) continue;
            this.snapshotCache.remove(key, cf);
            this.snapshotCacheExpiry.remove(key, last);
        }
    }

    private CompletableFuture<ChunkSnapshot> fetchChunkSnapshotAsync(World world, int cx, int cz) {
        ChunkKey key = new ChunkKey(world.getUID(), cx, cz);
        CompletableFuture<ChunkSnapshot> existing = this.snapshotCache.get(key);
        if (existing != null) {
            return existing;
        }
        CompletableFuture<ChunkSnapshot> future = new CompletableFuture<ChunkSnapshot>();
        CompletableFuture<ChunkSnapshot> raced = this.snapshotCache.putIfAbsent(key, future);
        if (raced != null) {
            return raced;
        }
        Messaging.debug("AsyncChunkCache: Fetching chunk", world, cx, cz);
        if (WORLD_GET_CHUNK_AT_ASYNC != null) {
            CompletableFuture chunkGetter;
            try {
                chunkGetter = WORLD_GET_CHUNK_AT_ASYNC.invoke(world, cx, cz, true, false);
            }
            catch (Throwable e) {
                future.completeExceptionally(e.getCause() != null ? e.getCause() : e);
                return future;
            }
            if (chunkGetter == null) {
                future.completeExceptionally(new IllegalStateException("getChunkAtAsync returned null"));
                return future;
            }
            chunkGetter.whenComplete((chunk, ex) -> {
                if (ex != null) {
                    future.completeExceptionally((Throwable)ex);
                    return;
                }
                try {
                    this.snapshotCacheExpiry.put(key, System.currentTimeMillis() + this.ttlMillis);
                    future.complete(chunk.getChunkSnapshot());
                }
                catch (Throwable t) {
                    future.completeExceptionally(t);
                }
            });
        } else {
            CitizensAPI.getScheduler().runRegionTask(world, cx, cz, () -> {
                try {
                    Chunk chunk = world.getChunkAt(cx, cz);
                    this.snapshotCacheExpiry.put(key, System.currentTimeMillis() + this.ttlMillis);
                    future.complete(chunk.getChunkSnapshot());
                }
                catch (Throwable t) {
                    future.completeExceptionally(t);
                }
            });
        }
        return future;
    }

    public CompletableFuture<Path> findPathAsync(PathRequest req) {
        ImmutableList rects;
        Rect end;
        Rect start = new Rect((req.from.getBlockX() >> 4) - req.prefetchRadius, (req.from.getBlockZ() >> 4) - req.prefetchRadius, (req.from.getBlockX() >> 4) + req.prefetchRadius, (req.from.getBlockZ() >> 4) + req.prefetchRadius);
        if (start.overlaps(end = new Rect((req.to.getBlockX() >> 4) - req.prefetchRadius, (req.to.getBlockZ() >> 4) - req.prefetchRadius, (req.to.getBlockX() >> 4) + req.prefetchRadius, (req.to.getBlockZ() >> 4) + req.prefetchRadius))) {
            Rect merged = new Rect(Math.min(start.minX, end.minX), Math.min(start.minZ, end.minZ), Math.max(start.maxX, end.maxX), Math.max(start.maxZ, end.maxZ));
            rects = ImmutableList.of((Object)merged);
        } else {
            rects = ImmutableList.of((Object)start, (Object)end);
        }
        for (Rect rect : rects) {
            for (int cx = rect.minX; cx <= rect.maxX; ++cx) {
                for (int cz = rect.minZ; cz <= rect.maxZ; ++cz) {
                    this.snapshotCache.computeIfAbsent(new ChunkKey(req.from.getWorld().getUID(), cx, cz), k -> new CompletableFuture());
                }
            }
        }
        CompletableFuture[] futures = new CompletableFuture[rects.size()];
        for (int i = 0; i < rects.size(); ++i) {
            futures[i] = this.prefetchRectangle(req.from.getWorld(), (Rect)rects.get(i));
        }
        CompletionStage workerFuture = CompletableFuture.allOf(futures).thenCompose(v -> CompletableFuture.supplyAsync(() -> this.runPathfinder(req, new SnapshotProvider(req.from.getWorld())), this.workerPool));
        CompletableFuture<Path> result = new CompletableFuture<Path>();
        ((CompletableFuture)workerFuture).whenComplete((res, ex) -> {
            Runnable cb = () -> {
                if (ex != null) {
                    result.completeExceptionally((Throwable)ex);
                } else {
                    result.complete((Path)res);
                }
            };
            if (Bukkit.isPrimaryThread()) {
                cb.run();
            } else {
                CitizensAPI.getScheduler().runTask(cb);
            }
        });
        return result;
    }

    private CompletableFuture<Void> prefetchIndividualChunks(World world, Rect rect) {
        ArrayList<CompletableFuture> chunkFutures = new ArrayList<CompletableFuture>();
        IdentityHashMap<CompletableFuture, ChunkKey> mapping = new IdentityHashMap<CompletableFuture, ChunkKey>();
        for (int cx = rect.minX; cx <= rect.maxX; ++cx) {
            for (int cz = rect.minZ; cz <= rect.maxZ; ++cz) {
                ChunkKey key = new ChunkKey(world.getUID(), cx, cz);
                CompletableFuture<ChunkSnapshot> snap = this.snapshotCache.get(key);
                if (snap != null && snap.isDone()) continue;
                try {
                    CompletableFuture chunkFuture = WORLD_GET_CHUNK_AT_ASYNC.invoke(world, cx, cz, true, false);
                    if (chunkFuture == null) {
                        if (snap == null || snap.isDone()) continue;
                        snap.completeExceptionally(new IllegalStateException("getChunkAtAsync returned null"));
                        continue;
                    }
                    chunkFutures.add(chunkFuture);
                    mapping.put(chunkFuture, key);
                    continue;
                }
                catch (Throwable e) {
                    if (snap == null || snap.isDone()) continue;
                    snap.completeExceptionally(e.getCause() != null ? e.getCause() : e);
                }
            }
        }
        if (chunkFutures.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        Messaging.debug("AsyncChunkCache: Fetching chunks", world, rect);
        CompletableFuture<Void> all = CompletableFuture.allOf(chunkFutures.toArray(new CompletableFuture[chunkFutures.size()]));
        CompletableFuture<Void> done = new CompletableFuture<Void>();
        all.whenComplete((v, ex) -> {
            for (CompletableFuture completed : chunkFutures) {
                CompletableFuture<ChunkSnapshot> pending;
                ChunkKey key = (ChunkKey)mapping.get(completed);
                if (key == null || (pending = this.snapshotCache.get(key)) == null || pending.isDone()) continue;
                try {
                    Chunk chunk = (Chunk)completed.join();
                    this.snapshotCacheExpiry.put(key, System.currentTimeMillis() + this.ttlMillis);
                    pending.complete(chunk.getChunkSnapshot());
                }
                catch (CompletionException ce) {
                    Throwable cause = ce.getCause() != null ? ce.getCause() : ce;
                    pending.completeExceptionally(cause);
                }
                catch (Throwable t) {
                    pending.completeExceptionally(t);
                }
            }
            done.complete(null);
        });
        return done;
    }

    private CompletableFuture<Void> prefetchRectangle(World world, Rect rect) {
        int chunksToLoad = 0;
        block4: for (int cx = rect.minX; cx <= rect.maxX; ++cx) {
            for (int cz = rect.minZ; cz <= rect.maxZ; ++cz) {
                ChunkKey key = new ChunkKey(world.getUID(), cx, cz);
                CompletableFuture<ChunkSnapshot> pending = this.snapshotCache.get(key);
                if (pending == null) {
                    throw new IllegalStateException();
                }
                if (pending.isDone()) continue;
                ++chunksToLoad;
                continue block4;
            }
        }
        if (chunksToLoad == 0) {
            return CompletableFuture.completedFuture(null);
        }
        Messaging.debug("AsyncChunkCache: Fetching chunk rectangle", world, rect);
        if (WORLD_GET_CHUNKS_AT_ASYNC != null) {
            CompletableFuture<Void> result = new CompletableFuture<Void>();
            Runnable callback = () -> {
                for (int cx = rect.minX; cx <= rect.maxX; ++cx) {
                    for (int cz = rect.minZ; cz <= rect.maxZ; ++cz) {
                        ChunkKey key = new ChunkKey(world.getUID(), cx, cz);
                        CompletableFuture<ChunkSnapshot> pending = this.snapshotCache.get(key);
                        if (pending == null || pending.isDone()) continue;
                        this.snapshotCacheExpiry.put(key, System.currentTimeMillis() + this.ttlMillis);
                        pending.complete(world.getChunkAt(cx, cz).getChunkSnapshot());
                    }
                }
                result.complete(null);
            };
            try {
                WORLD_GET_CHUNKS_AT_ASYNC.invoke(world, rect.minX, rect.minZ, rect.maxX, rect.maxZ, false, callback);
                return result;
            }
            catch (Throwable e) {
                e.printStackTrace();
            }
        }
        if (WORLD_GET_CHUNK_AT_ASYNC != null) {
            CompletableFuture<Void> result = new CompletableFuture<Void>();
            try {
                this.prefetchIndividualChunks(world, rect).whenComplete((v, ex) -> {
                    if (ex != null) {
                        result.completeExceptionally((Throwable)ex);
                    } else {
                        result.complete(null);
                    }
                });
            }
            catch (Throwable t) {
                result.completeExceptionally(t);
            }
            return result;
        }
        CompletableFuture<Void> result = new CompletableFuture<Void>();
        CitizensAPI.getScheduler().runRegionTask(world, rect.minX, rect.maxZ, () -> {
            try {
                for (int cx = rect.minX; cx <= rect.maxX; ++cx) {
                    for (int cz = rect.minZ; cz <= rect.maxZ; ++cz) {
                        ChunkKey key = new ChunkKey(world.getUID(), cx, cz);
                        CompletableFuture<ChunkSnapshot> pending = this.snapshotCache.get(key);
                        if (pending == null || pending.isDone()) continue;
                        this.snapshotCacheExpiry.put(key, System.currentTimeMillis());
                        pending.complete(world.getChunkAt(cx, cz).getChunkSnapshot());
                    }
                }
                result.complete(null);
            }
            catch (Throwable t) {
                result.completeExceptionally(t);
            }
        });
        return result;
    }

    private Path runPathfinder(final PathRequest req, final SnapshotProvider provider) {
        VectorGoal goal = new VectorGoal(req.to, (float)req.parameters.pathDistanceMargin());
        return (Path)AStarMachine.createWithDefaultStorage().runFully(goal, new VectorNode(goal, req.from, new BlockSource(){

            @Override
            public BlockData getBlockDataAt(int x, int y, int z) {
                return provider.get(x >> 4, z >> 4).getBlockData(x & 0xF, y, z & 0xF);
            }

            @Override
            public BoundingBox getCollisionBox(int x, int y, int z) {
                return null;
            }

            @Override
            public Material getMaterialAt(int x, int y, int z) {
                return provider.get(x >> 4, z >> 4).getBlockType(x & 0xF, y, z & 0xF);
            }

            @Override
            public boolean isYWithinBounds(int y) {
                return SpigotUtil.checkYSafe(y, req.from.getWorld());
            }
        }, req.parameters));
    }

    public void shutdown() {
        try {
            if (this.evictionExecutor != null) {
                this.evictionExecutor.shutdownNow();
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        this.snapshotCache.clear();
        this.snapshotCacheExpiry.clear();
    }

    static {
        try {
            WORLD_GET_CHUNKS_AT_ASYNC = MethodHandles.lookup().unreflect(World.class.getMethod("getChunksAtAsync", Integer.TYPE, Integer.TYPE, Integer.TYPE, Integer.TYPE, Boolean.TYPE, Runnable.class));
        }
        catch (Throwable ignored) {
            ignored.printStackTrace();
        }
        try {
            WORLD_GET_CHUNK_AT_ASYNC = MethodHandles.lookup().unreflect(World.class.getMethod("getChunkAtAsync", Integer.TYPE, Integer.TYPE, Boolean.TYPE, Boolean.TYPE));
        }
        catch (Throwable ignored) {
            ignored.printStackTrace();
        }
    }

    private static class ChunkKey {
        public final int cx;
        public final int cz;
        public final UUID uuid;

        public ChunkKey(UUID uuid, int cx, int cz) {
            this.uuid = uuid;
            this.cx = cx;
            this.cz = cz;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ChunkKey)) {
                return false;
            }
            ChunkKey k = (ChunkKey)o;
            return this.cx == k.cx && this.cz == k.cz && Objects.equals(this.uuid, k.uuid);
        }

        public int hashCode() {
            return Objects.hash(this.uuid, this.cx, this.cz);
        }

        public String toString() {
            return this.uuid + "@" + this.cx + "," + this.cz;
        }
    }

    private static class Rect {
        final int minX;
        final int minZ;
        final int maxX;
        final int maxZ;

        Rect(int minX, int minZ, int maxX, int maxZ) {
            this.minX = minX;
            this.minZ = minZ;
            this.maxX = maxX;
            this.maxZ = maxZ;
        }

        private boolean overlaps(Rect b) {
            return this.maxX >= b.minX && b.maxX >= this.minX && this.maxZ >= b.minZ && b.maxZ >= this.minZ;
        }

        public String toString() {
            return "[minX=" + this.minX + ", minZ=" + this.minZ + ", maxX=" + this.maxX + ", maxZ=" + this.maxZ + "]";
        }
    }

    public static final class PathRequest {
        private final Location from;
        private final NavigatorParameters parameters;
        private final int prefetchRadius;
        private final Location to;

        public PathRequest(Location from, Location to, int prefetchRadius, NavigatorParameters parameters) {
            this.from = from;
            this.to = to;
            this.parameters = parameters;
            this.prefetchRadius = prefetchRadius;
        }
    }

    private class SnapshotProvider {
        private final World world;

        SnapshotProvider(World world) {
            this.world = world;
        }

        public ChunkSnapshot get(int cx, int cz) {
            CompletableFuture<ChunkSnapshot> chunk = this.getAsync(cx, cz);
            if (!chunk.isDone() && Thread.currentThread() instanceof ForkJoinWorkerThread) {
                try {
                    ForkJoinPool.managedBlock(new CompletableFutureManagedBlocker<ChunkSnapshot>(chunk));
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new CompletionException(e);
                }
            }
            return chunk.join();
        }

        public CompletableFuture<ChunkSnapshot> getAsync(int cx, int cz) {
            return AsyncChunkCache.this.fetchChunkSnapshotAsync(this.world, cx, cz);
        }
    }

    private static class CompletableFutureManagedBlocker<T>
    implements ForkJoinPool.ManagedBlocker {
        private final CompletableFuture<T> cf;

        CompletableFutureManagedBlocker(CompletableFuture<T> cf) {
            this.cf = cf;
        }

        @Override
        public boolean block() throws InterruptedException {
            try {
                this.cf.join();
            }
            catch (CompletionException completionException) {
                // empty catch block
            }
            return true;
        }

        @Override
        public boolean isReleasable() {
            return this.cf.isDone();
        }
    }
}

