package net.citizensnpcs.api.hpastar;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;

import org.bukkit.Location;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

import ch.ethz.globis.phtree.PhTreeSolid;
import ch.ethz.globis.phtree.PhTreeSolid.PhQueryS;
import net.citizensnpcs.api.astar.Plan;
import net.citizensnpcs.api.astar.pathfinder.BlockSource;
import net.citizensnpcs.api.astar.pathfinder.MinecraftBlockExaminer;
import net.citizensnpcs.api.astar.pathfinder.Path;
import net.citizensnpcs.api.util.Messaging;

public class HPAGraph {
    private final BlockSource blockSource;
    public List<List<HPACluster>> clusters = new ArrayList<>();
    private final int cx, cy, cz;
    private final List<PhTreeSolid<HPACluster>> phtrees = new ArrayList<>();

    public HPAGraph(BlockSource blockSource, int cx, int cy, int cz) {
        this.blockSource = blockSource;
        this.cx = cx;
        this.cy = cy;
        this.cz = cz;

        while (clusters.size() <= MAX_DEPTH) {
            clusters.add(new ArrayList<>());
            if (clusters.size() != phtrees.size()) {
                phtrees.add(PhTreeSolid.create(3));
            }
        }
    }

    public void addClusters(int x, int z) {
        int baseX = MAX_CLUSTER_SIZE * ((x - cx) / MAX_CLUSTER_SIZE) + cx;
        int baseZ = MAX_CLUSTER_SIZE * ((z - cz) / MAX_CLUSTER_SIZE) + cz;

        if (phtrees.size() == 0) {
            phtrees.add(PhTreeSolid.create(3));
        }
        PhTreeSolid<HPACluster> baseLevel = phtrees.get(0);

        // Check if clusters already exist in this region
        PhQueryS<HPACluster> existingQuery = baseLevel.queryIntersect(new long[] { baseX, 0, baseZ },
                new long[] { baseX + MAX_CLUSTER_SIZE, 256, baseZ + MAX_CLUSTER_SIZE });
        if (existingQuery.hasNext()) {
            Messaging.log("Clusters already exist for region:", baseX, baseZ);
            return;
        }
        Messaging.log("Building clusters for:", baseX, baseZ);
        List<HPACluster> newClusters = new ArrayList<>();

        // build clusters
        int clusterSize = BASE_CLUSTER_SIZE;
        int clusterHeight = BASE_CLUSTER_HEIGHT;
        for (int y = 0; y < 256; y += clusterHeight) {
            for (int ci = 0; ci < MAX_CLUSTER_SIZE; ci += clusterSize) {
                for (int cj = 0; cj < MAX_CLUSTER_SIZE; cj += clusterSize) {
                    HPACluster cluster = new HPACluster(this, 0, clusterSize, clusterHeight, baseX + ci, y, baseZ + cj);
                    if (!cluster.hasWalkableNodes()) {
                        continue;
                    }
                    newClusters.add(cluster);
                    baseLevel.put(new long[] { cluster.clusterX, cluster.clusterY, cluster.clusterZ },
                            new long[] { cluster.clusterX + clusterSize, cluster.clusterY + clusterHeight,
                                    cluster.clusterZ + clusterSize },
                            cluster);
                    Messaging.log(cluster);
                }
            }
        }
        Multimap<HPACluster, HPACluster> neighbours = HashMultimap.create();
        for (HPACluster cluster : newClusters) {
            PhQueryS<HPACluster> q = baseLevel.queryIntersect(
                    new long[] { cluster.clusterX - clusterSize, cluster.clusterY - clusterHeight,
                            cluster.clusterZ - clusterSize },
                    new long[] { cluster.clusterX + clusterSize, cluster.clusterY + clusterHeight,
                            cluster.clusterZ + clusterSize });
            while (q.hasNext()) {
                HPACluster neighbour = q.nextValue();
                if (neighbour == cluster || neighbours.get(cluster).contains(neighbour)) {
                    continue;
                }
                int dx = neighbour.clusterX - cluster.clusterX;
                int dy = neighbour.clusterY - cluster.clusterY;
                int dz = neighbour.clusterZ - cluster.clusterZ;

                int nonZeroCount = (dx != 0 ? 1 : 0) + (dy != 0 ? 1 : 0) + (dz != 0 ? 1 : 0);
                if (nonZeroCount == 1) {
                    Direction direction = null;
                    if (dx > 0) {
                        direction = Direction.EAST;
                    } else if (dx < 0) {
                        direction = Direction.WEST;
                    } else if (dz > 0) {
                        direction = Direction.NORTH;
                    } else if (dz < 0) {
                        direction = Direction.SOUTH;
                    } else if (dy > 0) {
                        direction = Direction.UP;
                    } else if (dy < 0) {
                        direction = Direction.DOWN;
                    }
                    if (direction == null) {
                        continue;
                    }
                    cluster.connect(neighbour, direction);
                    neighbours.get(cluster).add(neighbour);
                    neighbours.get(neighbour).add(cluster);
                    Messaging.log("CONNECTED", cluster, neighbour);
                } else if (nonZeroCount == 2 && dy == 0) {
                    cluster.connectDiagonal(neighbour, Integer.signum(dx), Integer.signum(dz), DIAGONAL_WEIGHT);
                    neighbours.get(cluster).add(neighbour);
                    neighbours.get(neighbour).add(cluster);
                    Messaging.log("CONNECTED DIAGONAL", cluster, neighbour);
                }
            }
        }
        for (HPACluster cluster : newClusters) {
            cluster.connectIntra();
        }
        addClustersAtDepth(0, newClusters);
        for (int depth = 1; depth <= MAX_DEPTH; depth++) {
            newClusters = new ArrayList<>();
            clusterSize = (int) (BASE_CLUSTER_SIZE * Math.pow(2, depth));
            clusterHeight = (int) (BASE_CLUSTER_HEIGHT * Math.pow(2, depth));

            for (int y = 0; y < 256; y += clusterHeight) {
                for (int ci = 0; ci < MAX_CLUSTER_SIZE; ci += clusterSize) {
                    for (int cj = 0; cj < MAX_CLUSTER_SIZE; cj += clusterSize) {
                        HPACluster cluster = new HPACluster(this, depth, clusterSize, clusterHeight, baseX + ci, y,
                                baseZ + cj);
                        List<HPACluster> parentClusters = Lists.newArrayList(phtrees.get(depth - 1).queryInclude(
                                new long[] { cluster.clusterX, cluster.clusterY, cluster.clusterZ },
                                new long[] { cluster.clusterX + clusterSize, cluster.clusterY + clusterHeight,
                                        cluster.clusterZ + clusterSize }));
                        if (parentClusters.size() == 0) {
                            continue;
                        }
                        cluster.buildFrom(parentClusters);
                        phtrees.get(depth).put(new long[] { cluster.clusterX, cluster.clusterY, cluster.clusterZ },
                                new long[] { cluster.clusterX + clusterSize, cluster.clusterY + clusterHeight,
                                        cluster.clusterZ + clusterSize },
                                cluster);
                        Messaging.log(cluster);
                        newClusters.add(cluster);
                    }
                }
            }
            addClustersAtDepth(depth, newClusters);
        }
    }

    public void addClustersAtDepth(int depth, List<HPACluster> other) {
        clusters.get(depth).addAll(other);
    }

    public Plan findPath(Location start, Location goal) {
        List<HPACluster> clustersToClean = new ArrayList<>();
        HPAGraphNode startNode = new HPAGraphNode(start.getBlockX(), start.getBlockY(), start.getBlockZ()),
                goalNode = new HPAGraphNode(goal.getBlockX(), goal.getBlockY(), goal.getBlockZ());
        HPACluster startCluster = getClusterAt(0, startNode.x, startNode.y, startNode.z);
        HPACluster goalCluster = getClusterAt(0, goalNode.x, goalNode.y, goalNode.z);
        if (startCluster == null || goalCluster == null) {
            return new Path(Collections.emptyList());
        }
        startCluster.insert(startNode);
        clustersToClean.add(startCluster);
        if (goalCluster != startCluster) {
            goalCluster.insert(goalNode);
            clustersToClean.add(goalCluster);
        } else if (!goalNode.equals(startNode)) {
            startCluster.insert(goalNode);
        }
        AStarSolution sln = pathfind(startNode, goalNode, 0);
        System.out.println(":" + start + "->" + goal + "@" + sln.cost);
        for (HPACluster cluster : clustersToClean) {
            cluster.remove(startNode, goalNode);
        }
        return new Path(sln.convertToVectors());
    }

    public void invalidateCluster(int x, int y, int z) {
        for (int depth = 0; depth < phtrees.size(); depth++) {
            PhTreeSolid<HPACluster> phtree = phtrees.get(depth);
            PhQueryS<HPACluster> q = phtree.queryIntersect(new long[] { x, y, z }, new long[] { x, y, z });
            while (q.hasNext()) {
                HPACluster cluster = q.nextValue();
                phtree.remove(new long[] { cluster.clusterX, cluster.clusterY, cluster.clusterZ },
                        new long[] { cluster.clusterX + cluster.clusterSize, cluster.clusterY + cluster.clusterHeight,
                                cluster.clusterZ + cluster.clusterSize });
                clusters.get(depth).remove(cluster);
                Messaging.log("Invalidated cluster:", cluster);
            }
        }
        addClusters(x, z);
    }

    AStarSolution pathfind(HPAGraphNode start, HPAGraphNode dest, int level) {
        if (start.equals(dest))
            return new AStarSolution(Lists.newArrayList(new HPAGraphAStarNode(start, null)), 0);
        Map<ReversableAStarNode, Float> open = new HashMap<>();
        Map<ReversableAStarNode, Float> closed = new HashMap<>();
        Queue<ReversableAStarNode> frontier = new PriorityQueue<>();
        ReversableAStarNode startNode = new HPAGraphAStarNode(start, null);
        frontier.add(startNode);
        open.put(startNode, startNode.g);
        while (!frontier.isEmpty()) {
            HPAGraphAStarNode node = (HPAGraphAStarNode) frontier.poll();
            List<HPAGraphEdge> edges = node.node.getEdges(level);
            if (start != node.node) {
                closed.put(node, node.g);
            }
            open.remove(node);
            for (HPAGraphEdge edge : edges) {
                HPAGraphAStarNode neighbour = new HPAGraphAStarNode(edge.to, edge);
                if (closed.containsKey(neighbour)) {
                    continue;
                }
                neighbour.parent = node;
                neighbour.g = node.g + edge.weight;
                neighbour.h = (float) edge.to.distance(dest);
                if (edge.to.equals(dest))
                    return new AStarSolution(neighbour.reconstructSolution(), neighbour.g);
                if (open.containsKey(neighbour)) {
                    if (neighbour.g > open.get(neighbour)) {
                        continue;
                    }
                    frontier.remove(neighbour);
                }
                open.put(neighbour, neighbour.g);
                frontier.add(neighbour);
            }
        }
        return new AStarSolution(null, Float.POSITIVE_INFINITY);
    }

    public boolean walkable(int x, int y, int z) {
        if (y == 0)
            return false;
        return MinecraftBlockExaminer.canStandIn(blockSource.getMaterialAt(x, y, z))
                && MinecraftBlockExaminer.canStandIn(blockSource.getMaterialAt(x, y + 1, z))
                && MinecraftBlockExaminer.canStandOn(blockSource.getMaterialAt(x, y - 1, z));
    }

    private HPACluster getClusterAt(int depth, int x, int y, int z) {
        PhQueryS<HPACluster> q = phtrees.get(depth).queryIntersect(new long[] { x, y, z }, new long[] { x, y, z });
        return q.hasNext() ? q.next() : null;
    }

    private static int BASE_CLUSTER_HEIGHT = (int) (2 * Math.pow(2, 2));
    private static int BASE_CLUSTER_SIZE = (int) (2 * Math.pow(2, 3));
    private static final float DIAGONAL_WEIGHT = (float) Math.sqrt(2);
    private static int MAX_CLUSTER_SIZE = (int) (2 * Math.pow(2, 5));
    private static int MAX_DEPTH = 3;
}
