package net.citizensnpcs.api.util;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.configuration.file.YamlConfiguration;

import com.google.common.collect.Lists;
import com.google.common.collect.Queues;
import com.google.common.io.Files;
import com.google.common.primitives.Ints;

public class YamlStorage implements Storage {
    private final FileConfiguration config;
    private final File file;
    private final boolean transformLists;

    public YamlStorage(File file) {
        this(file, null);
    }

    public YamlStorage(File file, String header) {
        this(file, header, false);
    }

    public YamlStorage(File file, String header, boolean transformLists) {
        config = new YamlConfiguration();
        tryIncreaseMaxCodepoints(config);
        this.transformLists = transformLists;
        this.file = file;
        if (!file.exists()) {
            create();
            if (header != null) {
                config.options().header(header);
            }
            save();
        }
    }

    private void create() {
        try {
            Messaging.debug("Creating file: " + file.getName());
            file.getParentFile().mkdirs();
            file.createNewFile();
        } catch (IOException ex) {
            Messaging.severe("Could not create file: " + file.getName());
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null || getClass() != obj.getClass())
            return false;
        YamlStorage other = (YamlStorage) obj;
        return Objects.equals(file, other.file);
    }

    @Override
    public DataKey getKey(String root) {
        return new MemoryDataKey(config, root);
    }

    @Override
    public int hashCode() {
        return 31 + (file == null ? 0 : file.hashCode());
    }

    @Override
    public boolean load() {
        try {
            config.load(file);
            if (transformLists) {
                transformListsToMapsInConfig(config);
            }
            return true;
        } catch (Exception ex) {
            ex.printStackTrace();
            return false;
        }
    }

    @Override
    public void save() {
        YamlConfiguration copy = new YamlConfiguration();
        for (String key : config.getKeys(false)) {
            copy.set(key, config.get(key));
        }
        save(copy);
    }

    private void save(YamlConfiguration from) {
        try {
            Files.createParentDirs(file);
            File temporaryFile = File.createTempFile(file.getName(), null, file.getParentFile());
            temporaryFile.deleteOnExit();
            if (transformLists) {
                transformMapsToListsInConfig(from);
            }
            from.save(temporaryFile);
            file.delete();
            temporaryFile.renameTo(file);
            temporaryFile.delete();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    @Override
    public void saveAsync() {
        YamlConfiguration copy = new YamlConfiguration();
        for (String key : config.getKeys(false)) {
            copy.set(key, config.get(key));
        }
        ForkJoinPool.commonPool().submit(() -> save(copy));
    }

    @Override
    public String toString() {
        return "YamlStorage {file=" + file + "}";
    }

    private void transformListsToMapsInConfig(ConfigurationSection root) {
        List<ConfigurationSection> queue = Lists.newArrayList(root);
        while (queue.size() > 0) {
            ConfigurationSection section = queue.remove(queue.size() - 1);
            for (String key : section.getKeys(false)) {
                Object value = section.get(key);
                if (value instanceof Collection) {
                    ConfigurationSection synthetic = section.createSection(key);
                    int i = 0;
                    for (Iterator<?> itr = ((Collection<?>) value).iterator(); itr.hasNext();) {
                        Object next = itr.next();
                        if (next instanceof Map) {
                            queue.add(synthetic.createSection(Integer.toString(i++), (Map<?, ?>) next));
                        } else {
                            synthetic.set(Integer.toString(i++), next);
                        }
                    }
                } else if (value instanceof ConfigurationSection) {
                    queue.add((ConfigurationSection) value);
                }
            }
        }
    }

    private void transformMapsToListsInConfig(ConfigurationSection root) {
        Queue<ConfigurationSection> queue = Queues.newArrayDeque();
        queue.add(root);
        List<Tuple> convert = Lists.newArrayList();
        while (queue.size() > 0) {
            ConfigurationSection parent = queue.poll();

            for (String key : parent.getKeys(false)) {
                Object value = parent.get(key);
                if (value instanceof ConfigurationSection) {
                    queue.add((ConfigurationSection) value);
                    convert.add(new Tuple(parent, key));
                }
            }
        }
        outer: for (Tuple t : convert) {
            List<Integer> ints = t.parent.getConfigurationSection(t.key).getKeys(false).stream()
                    .map(i -> Ints.tryParse(i)).collect(Collectors.toList());
            if (ints.size() == 0)
                continue;

            for (int i = 0; i < ints.size(); i++) {
                if (ints.get(i) == null || ints.get(i) != i)
                    continue outer;
            }
            t.parent.set(t.key, ints.stream().map(i -> t.parent.getConfigurationSection(t.key).get(Integer.toString(i)))
                    .collect(Collectors.toList()));
        }
    }

    private void tryIncreaseMaxCodepoints(FileConfiguration config) {
        if (SET_CODEPOINT_LIMIT == null || LOADER_OPTIONS == null)
            return;
        try {
            SET_CODEPOINT_LIMIT.invoke(LOADER_OPTIONS.get(config), 67108864 /* ~64MB, Paper's limit */);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static class Tuple {
        String key;
        ConfigurationSection parent;

        public Tuple(ConfigurationSection parent2, String key2) {
            parent = parent2;
            key = key2;
        }
    }

    private static Field LOADER_OPTIONS;
    private static Method SET_CODEPOINT_LIMIT;
    static {
        try {
            LOADER_OPTIONS = YamlConfiguration.class.getDeclaredField("yamlLoaderOptions");
            LOADER_OPTIONS.setAccessible(true);
            SET_CODEPOINT_LIMIT = Class.forName("org.yaml.snakeyaml.LoaderOptions").getMethod("setCodepointLimit",
                    int.class);
            SET_CODEPOINT_LIMIT.setAccessible(true);
        } catch (Exception e) {
        }
    }
}