package net.citizensnpcs.api.astar.pathfinder;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.bukkit.Location;
import org.bukkit.util.BlockVector;
import org.bukkit.util.Vector;

import net.citizensnpcs.api.ai.NavigatorParameters;
import net.citizensnpcs.api.astar.AStarNode;
import net.citizensnpcs.api.astar.Plan;
import net.citizensnpcs.api.astar.pathfinder.BlockExaminer.AdditionalNeighbourGenerator;
import net.citizensnpcs.api.astar.pathfinder.BlockExaminer.PassableState;
import net.citizensnpcs.api.astar.pathfinder.BlockExaminer.ReplacementNeighbourGenerator;
import net.citizensnpcs.api.astar.pathfinder.BlockExaminer.StandableState;

public class VectorNode extends AStarNode implements PathPoint {
    private float blockCost = -1;
    List<PathCallback> callbacks;
    private final PathInfo info;
    Vector location;
    Vector locationCache;
    List<Vector> pathVectors;

    public VectorNode(VectorGoal goal, Location location, BlockSource source, NavigatorParameters params) {
        this(null, goal, location.toVector(), source, params);
    }

    public VectorNode(VectorNode parent, Vector location, PathInfo info) {
        super(parent);
        this.location = new BlockVector(location.getBlockX(), location.getBlockY(), location.getBlockZ());
        this.info = info;
    }

    public VectorNode(VectorNode parent, VectorGoal goal, Vector location, BlockSource source,
            NavigatorParameters params) {
        this(parent, location, new PathInfo(source, params, goal));
    }

    @Override
    public void addCallback(PathCallback callback) {
        if (callbacks == null) {
            callbacks = new ArrayList<>();
        }
        callbacks.add(callback);
    }

    @Override
    public Plan buildPlan() {
        return new Path(orderedPath(), info.goal.getGoalVector());
    }

    @Override
    public VectorNode createAtOffset(Vector mod) {
        return new VectorNode(this, mod, info);
    }

    @Override
    public PathPoint createAtOffset(Vector mod, float fixedCost) {
        VectorNode node = createAtOffset(mod);
        node.blockCost = node.getBlockCost() + fixedCost;
        return node;
    }

    public float distance(Vector goal) {
        int dx = Math.abs(location.getBlockX() - goal.getBlockX());
        int dy = Math.abs(location.getBlockY() - goal.getBlockY());
        int dz = Math.abs(location.getBlockZ() - goal.getBlockZ());

        return Math.max(dx, Math.max(dy, dz));
    }

    public float distance(VectorNode to) {
        return distance(to.location);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null || getClass() != obj.getClass())
            return false;
        VectorNode other = (VectorNode) obj;
        if (!Objects.equals(location, other.location))
            return false;
        return true;
    }

    private float getBlockCost() {
        if (blockCost == -1) {
            blockCost = 0;
            for (BlockExaminer examiner : info.params.examiners()) {
                blockCost += examiner.getCost(info.blockSource, this);
            }
        }
        return blockCost;
    }

    @Override
    public Vector getGoal() {
        return info.goal.getGoalVector();
    }

    @Override
    public Iterable<AStarNode> getNeighbours() {
        List<PathPoint> neighbours = null;
        for (BlockExaminer examiner : info.params.examiners()) {
            if (examiner instanceof ReplacementNeighbourGenerator) {
                neighbours = ((ReplacementNeighbourGenerator) examiner).getNeighbours(info.blockSource, this);
                break;
            }
        }
        if (neighbours == null) {
            neighbours = getNeighbours(info.blockSource, this);
        }
        for (BlockExaminer examiner : info.params.examiners()) {
            if (examiner instanceof AdditionalNeighbourGenerator) {
                ((AdditionalNeighbourGenerator) examiner).addNeighbours(info.blockSource, this, neighbours);
            }
        }
        List<AStarNode> nodes = new ArrayList<>(neighbours.size());
        for (PathPoint sub : neighbours) {
            if (!isPassable(sub))
                continue;

            nodes.add((AStarNode) sub);
        }
        return nodes;
    }

    public List<PathPoint> getNeighbours(BlockSource source, PathPoint point) {
        return getNeighbours(source, point, true);
    }

    public List<PathPoint> getNeighbours(BlockSource source, PathPoint point, boolean checkPassable) {
        List<PathPoint> neighbours = new ArrayList<>(25);
        for (int x = -1; x <= 1; x++) {
            for (int y = -1; y <= 1; y++) {
                for (int z = -1; z <= 1; z++) {
                    if (x == 0 && y == 0 && z == 0)
                        continue;

                    int modY = location.getBlockY() + y;
                    if (!source.isYWithinBounds(modY))
                        continue;

                    Vector mod = new Vector(location.getX() + x, modY, location.getZ() + z);
                    if (x != 0 && z != 0 && checkPassable) {
                        if (!isPassable(point.createAtOffset(new Vector(location.getX() + x, modY, location.getZ())))
                                || !isPassable(
                                        point.createAtOffset(new Vector(location.getX(), modY, location.getZ() + z))))
                            continue;
                    }
                    neighbours.add(point.createAtOffset(mod));
                }
            }
        }
        return neighbours;
    }

    @Override
    public PathPoint getParentPoint() {
        return (PathPoint) getParent();
    }

    @Override
    public List<Vector> getPathVectors() {
        return pathVectors;
    }

    @Override
    public Vector getVector() {
        if (locationCache == null) {
            locationCache = location.clone();
        }
        return locationCache.setX(location.getBlockX()).setY(location.getBlockY()).setZ(location.getBlockZ());
    }

    @Override
    public int hashCode() {
        return 31 + (location == null ? 0 : location.hashCode());
    }

    public float heuristicDistance(Vector goal) {
        return (distance(goal) + getBlockCost()) * TIEBREAKER;
    }

    private boolean isPassable(PathPoint mod) {
        boolean canStand = false;
        BlockExaminer found = null;
        for (BlockExaminer examiner : info.params.examiners()) {
            StandableState state = examiner.canStandAt(info.blockSource, mod);
            if (state == StandableState.STANDABLE) {
                canStand = true;

                // quickly check this examiner
                if (examiner.isPassable(info.blockSource, mod) == PassableState.PASSABLE)
                    return true;
                found = examiner;
                break;
            }
        }
        if (!canStand)
            return false;

        boolean passable = false;
        for (BlockExaminer examiner : info.params.examiners()) {
            if (examiner == found)
                continue;
            PassableState state = examiner.isPassable(info.blockSource, mod);
            if (state == PassableState.PASSABLE) {
                passable = true;
                break;
            }
        }
        return passable;
    }

    @Override
    public void setPathVectors(List<Vector> vectors) {
        this.pathVectors = vectors;
    }

    @Override
    public void setVector(Vector vector) {
        this.location = vector;
    }

    private static class PathInfo {
        private final BlockSource blockSource;
        private final VectorGoal goal;
        private final NavigatorParameters params;

        private PathInfo(BlockSource source, NavigatorParameters params, VectorGoal goal) {
            this.blockSource = source;
            this.params = params;
            this.goal = goal;
        }
    }

    private static final float TIEBREAKER = 1.01f;
}