package net.citizensnpcs.npc.skin;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.WeakHashMap;

import javax.annotation.Nullable;

import org.bukkit.ChatColor;

import com.mojang.authlib.GameProfile;

import net.citizensnpcs.Settings.Setting;
import net.citizensnpcs.api.CitizensAPI;
import net.citizensnpcs.api.event.DespawnReason;
import net.citizensnpcs.api.event.SpawnReason;
import net.citizensnpcs.api.npc.NPC;
import net.citizensnpcs.api.util.Messaging;
import net.citizensnpcs.npc.skin.profile.ProfileFetcher;
import net.citizensnpcs.trait.SkinTrait;
import net.citizensnpcs.util.GameProfileWrapper;
import net.citizensnpcs.util.SkinProperty;

/**
 * Stores data for a single skin.
 */
public class Skin {
    private boolean fetching;
    private int fetchRetries = -1;
    private boolean hasFetched;
    private volatile boolean isValid = true;
    private final Map<SkinnableEntity, Void> pending = new WeakHashMap<>(15);
    private net.citizensnpcs.api.util.schedulers.SchedulerTask retryTask;
    private volatile SkinProperty skinData;
    private volatile UUID skinId;
    private final String skinName;

    /**
     * Constructor.
     *
     * @param skinName
     *            The name of the player the skin belongs to.
     */
    Skin(String skinName) {
        this.skinName = skinName.toLowerCase(Locale.ROOT);

        synchronized (CACHE) {
            if (CACHE.containsKey(this.skinName))
                throw new IllegalArgumentException("There is already a skin named " + skinName);

            CACHE.put(this.skinName, this);
        }
        // fetch();
    }

    /**
     * Apply the skin data to the specified skinnable entity.
     *
     * <p>
     * If invoked before the skin data is ready, the skin is retrieved and the skin is automatically applied to the
     * entity at a later time.
     * </p>
     *
     * @param entity
     *            The skinnable entity.
     *
     * @return True if skin was applied, false if the data is being retrieved.
     */
    public boolean apply(SkinnableEntity entity) {
        Objects.requireNonNull(entity);

        NPC npc = entity.getNPC();
        SkinTrait skinTrait = npc.getOrAddTrait(SkinTrait.class);
        // Use npc cached skin if available.
        // If npc requires latest skin, cache is used for faster availability until the latest skin can be loaded.
        String cachedName = npc.data().get(CACHED_SKIN_UUID_NAME_METADATA);
        String texture = skinTrait.getTexture();
        if (skinName.equals(cachedName) && texture != null && !texture.equals("cache")) {
            setNPCTexture(entity, new SkinProperty("textures", texture, skinTrait.getSignature()));

            if (!skinTrait.shouldUpdateSkins()) // cache preferred
                return true;
        }
        if (!hasSkinData()) {
            String npcName = ChatColor.stripColor(npc.getName()).toLowerCase(Locale.ROOT);

            if (!skinTrait.shouldUpdateSkins() && !skinTrait.fetchDefaultSkin() && skinName.equals(npcName))
                return false;

            if (hasFetched)
                return true;

            if (!fetching) {
                fetch();
            }
            pending.put(entity, null);
            return false;
        }
        setNPCSkinData(entity, skinName, skinId, skinData);

        return true;
    }

    /**
     * Apply the skin data to the specified skinnable entity and respawn the NPC.
     *
     * @param entity
     *            The skinnable entity.
     */
    public void applyAndRespawn(SkinnableEntity entity) {
        Objects.requireNonNull(entity);

        if (!apply(entity))
            return;

        NPC npc = entity.getNPC();

        if (!npc.isSpawned())
            return;

        CitizensAPI.getScheduler().runEntityTask(npc.getEntity(), () -> {
            npc.despawn(DespawnReason.PENDING_RESPAWN);
            npc.spawn(npc.getStoredLocation(), SpawnReason.RESPAWN);
        });
    }

    private void fetch() {
        int maxRetries = Setting.MAX_NPC_SKIN_RETRIES.asInt();
        if (maxRetries > -1 && fetchRetries >= maxRetries) {
            Messaging.idebug(() -> "Reached max skin fetch retries for '" + skinName + "'");
            return;
        }
        if (skinName.length() < 3 || skinName.length() > 16) {
            return;
        }
        if (skinName.toLowerCase(Locale.ROOT).startsWith("cit-"))
            return;

        fetching = true;

        ProfileFetcher.fetch(skinName, request -> {
            hasFetched = true;

            switch (request.getResult()) {
                case NOT_FOUND:
                    isValid = false;
                    break;
                case TOO_MANY_REQUESTS:
                    if (maxRetries == 0)
                        break;

                    fetchRetries++;
                    long delay = Setting.NPC_SKIN_RETRY_DELAY.asTicks();
                    retryTask = CitizensAPI.getScheduler().runTaskLater(this::fetch, delay);

                    Messaging.idebug(() -> "Retrying skin fetch for '" + skinName + "' in " + delay + " ticks.");
                    break;
                case SUCCESS:
                    GameProfile profile = request.getProfile();
                    setData(profile);
                    break;
                default:
                    break;
            }
        });
    }

    private void fetchForced() {
        int maxRetries = Setting.MAX_NPC_SKIN_RETRIES.asInt();
        if (maxRetries > -1 && fetchRetries >= maxRetries) {
            Messaging.idebug(() -> "Reached max skin fetch retries for '" + skinName + "'");
            return;
        }
        if (skinName.length() < 3 || skinName.length() > 16)
            return;

        if (skinName.toLowerCase(Locale.ROOT).startsWith("cit-"))
            return;

        fetching = true;

        ProfileFetcher.fetchForced(skinName, request -> {
            hasFetched = true;

            switch (request.getResult()) {
                case NOT_FOUND:
                    isValid = false;
                    break;
                case TOO_MANY_REQUESTS:
                    if (maxRetries == 0) {
                        break;
                    }
                    fetchRetries++;
                    int delay = Setting.NPC_SKIN_RETRY_DELAY.asTicks();
                    retryTask = CitizensAPI.getScheduler().runTaskLater(this::fetchForced, delay);

                    Messaging.idebug(() -> "Retrying skin fetch for '" + skinName + "' in " + delay + " ticks.");
                    break;
                case SUCCESS:
                    GameProfile profile = request.getProfile();
                    setData(profile);
                    break;
                default:
                    break;
            }
        });
    }

    /**
     * Get the ID of the player the skin belongs to.
     *
     * @return The skin ID or null if it has not been retrieved yet or the skin is invalid.
     */
    @Nullable
    public UUID getSkinId() {
        return skinId;
    }

    /**
     * Get the name of the skin.
     */
    public String getSkinName() {
        return skinName;
    }

    /**
     * Determine if the skin data has been retrieved.
     */
    public boolean hasSkinData() {
        return skinData != null;
    }

    /**
     * Determine if the skin is valid.
     */
    public boolean isValid() {
        return isValid;
    }

    private void setData(@Nullable GameProfile profile) {
        if (profile == null) {
            isValid = false;
            return;
        }
        GameProfileWrapper gpw = GameProfileWrapper.fromMojangProfile(profile);
        if (!gpw.name.toLowerCase(Locale.ROOT).equals(skinName)) {
            Messaging.debug("GameProfile name (" + gpw.name + ") and " + "skin name (" + skinName
                    + ") do not match. Has the user renamed recently?");
        }
        skinId = gpw.uuid;
        skinData = SkinProperty.fromMojangProfile(profile);

        List<SkinnableEntity> entities = new ArrayList<>(pending.keySet());
        for (SkinnableEntity entity : entities) {
            applyAndRespawn(entity);
        }
        pending.clear();
    }

    /**
     * Clear all cached skins.
     */
    public static void clearCache() {
        synchronized (CACHE) {
            for (Skin skin : CACHE.values()) {
                skin.pending.clear();
                if (skin.retryTask != null) {
                    skin.retryTask.cancel();
                }
            }
            CACHE.clear();
        }
    }

    /**
     * Get a skin for a skinnable entity.
     *
     * <p>
     * If a Skin instance does not exist, a new one is created and the skin data is automatically fetched.
     * </p>
     *
     * @param entity
     *            The skinnable entity.
     */
    public static Skin get(SkinnableEntity entity) {
        return get(entity, false);
    }

    /**
     * Get a skin for a skinnable entity.
     *
     * <p>
     * If a Skin instance does not exist, a new one is created and the skin data is automatically fetched.
     * </p>
     *
     * @param entity
     *            The skinnable entity.
     * @param forceUpdate
     *            if the skin should be checked via the cache
     */
    public static Skin get(SkinnableEntity entity, boolean forceUpdate) {
        Objects.requireNonNull(entity);

        return get(entity.getSkinName(), forceUpdate);
    }

    /**
     * Get a player skin.
     *
     * <p>
     * If a Skin instance does not exist, a new one is created and the skin data is automatically fetched.
     * </p>
     *
     * @param skinName
     *            The name of the skin.
     */
    public static Skin get(String skinName, boolean forceUpdate) {
        Objects.requireNonNull(skinName);

        skinName = skinName.toLowerCase(Locale.ROOT);

        Skin skin;
        synchronized (CACHE) {
            skin = CACHE.get(skinName);
        }
        if (skin == null) {
            skin = new Skin(skinName);
        } else if (forceUpdate) {
            skin.fetchForced();
        }
        return skin;
    }

    public static boolean hasSkin(String name) {
        synchronized (CACHE) {
            return CACHE.containsKey(name);
        }
    }

    private static void setNPCSkinData(SkinnableEntity entity, String skinName, UUID skinId,
            SkinProperty skinProperty) {
        NPC npc = entity.getNPC();
        SkinTrait skinTrait = npc.getOrAddTrait(SkinTrait.class);

        // cache skins for faster initial skin availability and for use when the latest skin is not required.
        npc.data().setPersistent(CACHED_SKIN_UUID_NAME_METADATA, skinName);
        npc.data().setPersistent(CACHED_SKIN_UUID_METADATA, skinId.toString());
        if (skinProperty.value != null) {
            skinTrait.applyTextureInternal(skinProperty.signature == null ? "" : skinProperty.signature,
                    skinProperty.value);
            setNPCTexture(entity, skinProperty);
        } else {
            skinTrait.clearTexture();
        }
    }

    private static void setNPCTexture(SkinnableEntity entity, SkinProperty skinProperty) {
        GameProfile profile = entity.gameProfile();

        // don't set property if already set since this sometimes causes packet errors that disconnect the client.
        SkinProperty current = SkinProperty.fromMojangProfile(profile);
        if (current != null && current.value.equals(skinProperty.value) && current.signature != null
                && current.signature.equals(skinProperty.signature))
            return;

        entity.applyTexture(skinProperty);
    }

    private static final Map<String, Skin> CACHE = new HashMap<>(20);
    public static final String CACHED_SKIN_UUID_METADATA = "cached-skin-uuid";
    public static final String CACHED_SKIN_UUID_NAME_METADATA = "cached-skin-uuid-name";
}
